/*:
 * @target MZ
 * @plugindesc ADV風バックログ＋ボイス再生（SimpleVoice連携） v1.0.0
 * @author Human Simulate × ChatGPT
 * @url https://example.invalid/hs_backlog_voice
 * @help
 * ツクールMZ用。会話テキストをバックログとして保存し、
 * バックログ上で OK（決定） すると、その行に紐づくボイスを再生します。
 * Triacontane様の「SimpleVoice.js」のボイス音量に連動します。
 *
 * ■できること
 * - Show Text の内容を自動で記録（名前ウィンドウにも対応）
 * - バックログ画面（スクロール可能）で過去メッセージを閲覧
 * - バックログ上で決定：その行のボイスを再生
 *   （SimpleVoice.js が導入されている場合は AudioManager.playVoice を使用し、
 *     “ボイス音量”に連動します。未導入時はSE再生で代用可）
 * - 右クリック/キャンセルで閉じる
 *
 * ■使い方（最小構成）
 * 1) このプラグインを SimpleVoice.js の *下* に配置して有効化してください。
 * 2) （任意）「オープンキー」を好みのキーに設定。
 * 3) イベントでは、テキストの直前に SimpleVoice の「PLAY_VOICE」コマンドで
 *    ボイスを鳴らすだけでOK。直後のテキストがそのボイスと紐づきます。
 * 4) ゲーム中に設定した「オープンキー」を押すとバックログが開きます。
 *
 * 例）
 *   ◆プラグインコマンド：SimpleVoice:PLAY_VOICE（name: voice/yukino_001, channel: 1）
 *   ◆文章の表示：\\N<雪乃>「……おはようございます」
 * → バックログでこの行を選んで決定すると yukino_001 が再生されます。
 *
 * ■注意
 * - テキストに含まれる制御文字のうち、\\V[n], \\N[n], \\P[n] は展開します。
 *   それ以外（\\C[n] など演出系）はバックログでは除去します。
 * - 行ごとに紐づけられるボイスは「その行の直前に最後に再生されたボイス」です。
 *   一つのメッセージに複数ボイスがある高度な演出には完全対応しません。
 *
 * @param maxEntries
 * @text 最大記録数
 * @type number
 * @min 50
 * @max 2000
 * @default 300
 *
 * @param openKeySymbol
 * @text オープンキー（Input.symbol）
 * @type select
 * @option pageup
 * @option pagedown
 * @option shift
 * @option control
 * @option tab
 * @option cancel
 * @option menu
 * @default pagedown
 *
 * @param captureDuringMessageOnly
 * @text 文章表示のみ記録
 * @type boolean
 * @on はい
 * @off いいえ
 * @default true
 *
 * @param showNameInList
 * @text 名前を先頭に表示
 * @type boolean
 * @on はい
 * @off いいえ
 * @default true
 *
 * @command OpenBacklog
 * @text バックログを開く
 * @desc バックログシーンを開きます。
 *
 * @command ClearBacklog
 * @text バックログを消去
 * @desc 現在のバックログをすべて消去します。
 */

(() => {
  'use strict';

  const PLUGIN_NAME = 'HS_BacklogVoice';
  const params = PluginManager.parameters(PLUGIN_NAME);
  const MAX_ENTRIES = Number(params.maxEntries || 300);
  const OPEN_KEY = String(params.openKeySymbol || 'pagedown');
  const CAPTURE_MESSAGE_ONLY = params.captureDuringMessageOnly === 'true';
  const SHOW_NAME = params.showNameInList === 'true';

  //---------------------------------------------------------------------------
  // 小ユーティリティ
  //---------------------------------------------------------------------------
  function toNumber(x, def){ const n = Number(x); return isNaN(n) ? def : n; }

  function convertBasicEscapesForBacklog(text) {
    // \\V[n]
    text = text.replace(/\\\\V\\[(\\d+)\\]/gi, (_, n) => String($gameVariables.value(Number(n)) ?? 0));
    // \\N[n] : actor name
    text = text.replace(/\\\\N\\[(\\d+)\\]/gi, (_, n) => {
      const a = $gameActors.actor(Number(n));
      return a ? a.name() : '';
    });
    // \\P[n] : party member
    text = text.replace(/\\\\P\\[(\\d+)\\]/gi, (_, n) => {
      const m = $gameParty.members()[Number(n) - 1];
      return m ? m.name() : '';
    });
    // 演出系を除去
    text = text
      .replace(/\\\\C\\[\\d+\\]/gi, '')
      .replace(/\\\\I\\[\\d+\\]/gi, '')
      .replace(/\\\\FS\\[\\d+\\]/gi, '')
      .replace(/\\\\\\\\/g, '\\\\') // 末端のエスケープを整える
      .replace(/\\\\\\{|\\\\\\}|\\\\\\.|\\\\\\||\\\\\\!|\\\\\\>|\\\\\\<|\\\\\\^|\\\\\\#|\\\\G|\\\\W\\[\\d+\\]/gi, '');
    return text;
  }

  function lastPlayedVoicePath() {
    return AudioManager._lastVoicePathForBacklog || null;
  }

  //---------------------------------------------------------------------------
  // Game_System: バックログ配列の保持
  //---------------------------------------------------------------------------
  const _Game_System_initialize = Game_System.prototype.initialize;
  Game_System.prototype.initialize = function() {
    _Game_System_initialize.apply(this, arguments);
    this._hsBacklog = [];
  };

  Game_System.prototype.hsPushBacklog = function(entry) {
    this._hsBacklog = this._hsBacklog || [];
    this._hsBacklog.push(entry);
    if (this._hsBacklog.length > MAX_ENTRIES) {
      this._hsBacklog.shift();
    }
  };

  Game_System.prototype.hsClearBacklog = function() {
    this._hsBacklog = [];
  };

  Game_System.prototype.hsBacklog = function() {
    return this._hsBacklog || [];
  };

  //---------------------------------------------------------------------------
  // SimpleVoice連携：再生された最後のボイスのパスを記録
  // （SimpleVoice がなくても安全に動くようにラップ）
  //---------------------------------------------------------------------------
  if (AudioManager.playVoice) {
    const _playVoice = AudioManager.playVoice;
    AudioManager.playVoice = function(voice, loop, channel) {
      const result = _playVoice.apply(this, arguments);
      try {
        if (voice && voice.name) {
          // SimpleVoiceは 'se/' を内部で付けるので、ここでは相対パス（例: voice/yukino_001）を保持
          this._lastVoicePathForBacklog = String(voice.name);
        }
      } catch (e) {
        // no-op
      }
      return result;
    };
  } else {
    // 非導入環境向け：SE再生をラップして代用（音量はSEになります）
    const _playSe = AudioManager.playSe;
    AudioManager.playSe = function(se) {
      const result = _playSe.apply(this, arguments);
      try { if (se && se.name) this._lastVoicePathForBacklog = String(se.name); } catch(e){}
      return result;
    };
  }

  //---------------------------------------------------------------------------
  // Window_Message: メッセージ開始時にログへ保存
  //---------------------------------------------------------------------------
  const _Window_Message_startMessage = Window_Message.prototype.startMessage;
  Window_Message.prototype.startMessage = function() {
    _Window_Message_startMessage.apply(this, arguments);
    try {
      const faceName  = $gameMessage.faceName ? $gameMessage.faceName() : $gameMessage._faceName;
      const faceIndex = $gameMessage.faceIndex ? $gameMessage.faceIndex() : $gameMessage._faceIndex;
      const speaker   = ($gameMessage.speakerName && $gameMessage.speakerName()) || $gameMessage._speakerName || '';
      const raw = ($gameMessage.allText && $gameMessage.allText()) || ($gameMessage._texts ? $gameMessage._texts.join('\\n') : '');
      const text = convertBasicEscapesForBacklog(raw);
      if (!text) return;
      // 文章表示中のみ記録する設定がONなら、このタイミングなので常に記録
      const entry = {
        speaker: speaker || '',
        text,
        faceName: faceName || '',
        faceIndex: Number.isFinite(faceIndex) ? faceIndex : 0,
        voicePath: lastPlayedVoicePath() // 同フレーム直前のボイスを関連付け
      };
      $gameSystem.hsPushBacklog(entry);
    } catch (e) {
      console.warn('[HS_BacklogVoice] push failed:', e);
    }
  };

  //---------------------------------------------------------------------------
  // Scene_Map / Scene_Message: オープンキーでバックログを開く
  //---------------------------------------------------------------------------
  function hsUpdateOpenKey(scene) {
    if (Input.isTriggered(OPEN_KEY)) {
      SceneManager.push(Scene_HS_Backlog);
    }
  }
  const _Scene_Map_update = Scene_Map.prototype.update;
  Scene_Map.prototype.update = function() {
    _Scene_Map_update.apply(this, arguments);
    hsUpdateOpenKey(this);
  };
  const _Scene_Message_update = Scene_Message.prototype.update;
  Scene_Message.prototype.update = function() {
    _Scene_Message_update.apply(this, arguments);
    hsUpdateOpenKey(this);
  };

  //---------------------------------------------------------------------------
  // プラグインコマンド
  //---------------------------------------------------------------------------
  PluginManager.registerCommand(PLUGIN_NAME, 'OpenBacklog', () => {
    SceneManager.push(Scene_HS_Backlog);
  });
  PluginManager.registerCommand(PLUGIN_NAME, 'ClearBacklog', () => {
    $gameSystem.hsClearBacklog();
  });

  //---------------------------------------------------------------------------
  // Window_HS_BacklogList: ログ一覧
  //---------------------------------------------------------------------------
  function Window_HS_BacklogList() {
    this.initialize(...arguments);
  }
  Window_HS_BacklogList.prototype = Object.create(Window_Selectable.prototype);
  Window_HS_BacklogList.prototype.constructor = Window_HS_BacklogList;

  Window_HS_BacklogList.prototype.initialize = function(rect) {
    Window_Selectable.prototype.initialize.call(this, rect);
    this._data = $gameSystem.hsBacklog().slice();
    this.activate();
    this.select(Math.max(this._data.length - 1, 0)); // 末尾を選択
  };

  Window_HS_BacklogList.prototype.maxItems = function() {
    return this._data.length;
  };

  Window_HS_BacklogList.prototype.item = function(index) {
    return this._data[index];
  };

  Window_HS_BacklogList.prototype.drawItem = function(index) {
    const rect = this.itemLineRect(index);
    const entry = this.item(index);
    if (!entry) return;
    const name = SHOW_NAME && entry.speaker ? `【${entry.speaker}】` : '';
    const text = name ? `${name} ${entry.text}` : entry.text;
    this.resetFontSettings();
    this.drawTextEx(text, rect.x, rect.y, rect.width);
    // ボイスアイコン（●）
    if (entry.voicePath) {
      this.changeTextColor(ColorManager.systemColor());
      this.drawText('●', rect.x + rect.width - this.textWidth('●') - 8, rect.y, 32, 'right');
      this.resetTextColor();
    }
  };

  Window_HS_BacklogList.prototype.itemHeight = function() {
    // 1行の高さ
    return this.lineHeight() * 3; // ざっくり3行分の高さ確保（簡易）
  };

  Window_HS_BacklogList.prototype.playEntryVoice = function() {
    const entry = this.item(this.index());
    if (!entry || !entry.voicePath) return;
    if (AudioManager.playVoice) {
      AudioManager.playVoice({ name: entry.voicePath, volume: 100, pitch: 100, pan: 0 }, false, 0);
    } else {
      // フォールバック（SE扱い）
      AudioManager.playSe({ name: entry.voicePath, volume: 90, pitch: 100, pan: 0 });
    }
  };

  Window_HS_BacklogList.prototype.processOk = function() {
    Window_Selectable.prototype.processOk.call(this);
    this.playEntryVoice();
  };

  //---------------------------------------------------------------------------
  // Scene_HS_Backlog
  //---------------------------------------------------------------------------
  function Scene_HS_Backlog() { this.initialize(...arguments); }
  Scene_HS_Backlog.prototype = Object.create(Scene_MenuBase.prototype);
  Scene_HS_Backlog.prototype.constructor = Scene_HS_Backlog;

  Scene_HS_Backlog.prototype.initialize = function() {
    Scene_MenuBase.prototype.initialize.call(this);
  };

  Scene_HS_Backlog.prototype.create = function() {
    Scene_MenuBase.prototype.create.call(this);
    const rect = this.backlogWindowRect();
    this._listWindow = new Window_HS_BacklogList(rect);
    this._listWindow.setHandler('cancel', this.popScene.bind(this));
    this.addWindow(this._listWindow);

    // ヘルプ（簡易）
    const h = new Window_Help(new Rectangle(rect.x, 0, rect.width, this.calcWindowHeight(2, false)));
    h.setText('↑↓選択 / 決定: ボイス再生 / 取消: 閉じる');
    this.addWindow(h);
    this._helpWindow = h;
  };

  Scene_HS_Backlog.prototype.backlogWindowRect = function() {
    const wx = 0;
    const wy = this.calcWindowHeight(2, false);
    const ww = Graphics.boxWidth;
    const wh = Graphics.boxHeight - wy;
    return new Rectangle(wx, wy, ww, wh);
  };

})();